/** <!------------------------------------------------------------------------->
*
*  @file PduR.c
*
*  @creationDate 22 November 2024 11:42:31
*
*  @brief Contains implementation of all the APIs for PDU router module
*
*  @author
*    SajuJ,Shanavas
*
*  @description
*    The PDU Router module provides services for routing of I-PDUs between
*    -communication interface modules like CanIf and LinIf
*    -Transport Protocol modules like CanTP
*    -DCM and Transport Protocol modules
*    -AUTOSAR COM and communication interface modules
*    -I-PDU Multiplexer and communication interface modules
*
*  @copyright
*    Copyright 2024, dSPACE GmbH. All rights reserved.
* 
*<!-------------------------------------------------------------------------->*/

/*----------------------------------------------------------------------------*/
/* INCLUDES */
/*----------------------------------------------------------------------------*/
#include "PduR.h"

#if (PDUR_CANIF_SUPPORT == STD_ON)
#include "PduR_CanIf.h"
#endif
#if (PDUR_LINIF_SUPPORT == STD_ON)
#include "PduR_LinIf.h"
#endif
#if (PDUR_COM_SUPPORT == STD_ON)
#include "PduR_Com.h"
#endif
#if (PDUR_IPDUM_SUPPORT == STD_ON)
#include "PduR_IpduM.h"
#endif
#if (PDUR_SECOC_SUPPORT == STD_ON)
#include "PduR_SecOC.h"
#endif
#if (PDUR_SOAD_SUPPORT == STD_ON)
#include "PduR_SoAd.h"
#endif
#if (PDUR_J1939TP_SUPPORT == STD_ON)
#include "PduR_J1939Tp.h"
#include "Com_generated.h"
#endif
#if (PDUR_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif
#ifdef PDUR_ENABLE_PROFILING
#include <DsApplicationInterface.h>
#endif


/*----------------------------------------------------------------------------*/
/* DEFINES / ENUMS */
/*----------------------------------------------------------------------------*/

/* PDUR336 */
#if (PDUR_ZERO_COST_OPERATION == STD_OFF)

/* PDU Router module version information. */
#define PDUR_MODULE_ID                      ((uint16)51)
#define PDUR_VENDOR_ID                      0x23 // 35
#define PDUR_INSTANCE_ID                    0x0
#define PDUR_MAJOR_VERSION                  0x03
#define PDUR_MINOR_VERSION                  0x02
#define PDUR_PATCH_VERSION                  0x01

/*----------------------------------------------------------------------------*/
/* GLOBAL VARIABLES */
/*----------------------------------------------------------------------------*/

#ifdef PDUR_ENABLE_PROFILING

/** Global variable list of dSPACE profiling sources per module. 
    Each module has 4 sources, one per callback: transmit, triggerTransmit, txConfirmation (not used) and rxIndication  */
static DsTProfilerSource g_PduRProfilingSource [3 * 2 ];
static DsTProfilerSource PduR_Transmit_PROFILER_SRC;
#endif


/* PDUR325:Variable indicates the state of the PDUR module  */
static PduR_StateType g_PduRState = PDUR_UNINIT;

/** Global variable points to the configuration structure autogenerated by the PduRGenerator */
static const PduR_PBConfigType* gp_PduRConfig = &PduR_Config;


/*----------------------------------------------------------------------------*/
/* GLOBAL FUNCTIONS */
/*----------------------------------------------------------------------------*/

void PduR_GetVersionInfo(Std_VersionInfoType* versionInfo)
{
    if (versionInfo != NULL_PTR)
    {
        versionInfo->vendorID = PDUR_VENDOR_ID;
        versionInfo->instanceID = PDUR_INSTANCE_ID;
        versionInfo->moduleID = PDUR_MODULE_ID;
        versionInfo->sw_major_version = PDUR_MAJOR_VERSION;
        versionInfo->sw_minor_version = PDUR_MINOR_VERSION;
        versionInfo->sw_patch_version = PDUR_PATCH_VERSION;
    }
}

/** <!------------------------------------------------------------------------->
*   Initializes the PDU Router module
*
*   @description Initializes all PDU transmit buffers and set the state of the 
*                module.
*
*
*   @parameters
*       @param ConfigPtr: Pointer to configuration parameter set
*
*   @return None
*
*   @note
*        PDUR106:Not implemented since minimum routing not supported
*        (also AR Bug:35676)
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_Init(const PduR_PBConfigType* ConfigPtr)
{
#if (PDUR_DEV_ERROR_DETECT == STD_ON)
    /* Set the return value as E_OK */
     Std_ReturnType ReturnValue = E_OK;
    /* PDUR222 */
    if (ConfigPtr == NULL_PTR)
    {
        /* Report DET error for a null pointer input  */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID_INIT,
                         PDUR_E_CONFIG_PTR_INVALID);
        ReturnValue = E_NOT_OK;
    }
    /* PDUR337 */
    if (g_PduRState != PDUR_UNINIT)
    {
        /* Module already initialized */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID_INIT,
                         PDUR_E_INVALID_REQUEST);
        ReturnValue = E_NOT_OK;
    }
    if (ReturnValue == E_OK)
#endif
#ifdef PDUR_ENABLE_PROFILING
	{
		g_PduRProfilingSource[0] = dsProfilerCreateSource(MACRO_NAME_TOSTR(ECU_ID)"_CanIf_Transmit");
		g_PduRProfilingSource[0 + 1] = 0;
		g_PduRProfilingSource[0 + 2] = 0;

		g_PduRProfilingSource[3] = 0;
		g_PduRProfilingSource[3 + 1] = dsProfilerCreateSource(MACRO_NAME_TOSTR(ECU_ID)"_Com_TriggerTransmit");
		g_PduRProfilingSource[3 + 2] = dsProfilerCreateSource(MACRO_NAME_TOSTR(ECU_ID)"_Com_RxIndication");

        PduR_Transmit_PROFILER_SRC = dsProfilerCreateSource(MACRO_NAME_TOSTR(ECU_ID)"_PduR_Transmit");
    }
#endif
    {
        /* PDUR326 */
        g_PduRState = PDUR_ONLINE;
    }
    INTENDED_UNREFERENCED_PARAMETER(ConfigPtr);
}

void PduR_VariableInitialize(void)
{
}

/** <!------------------------------------------------------------------------->
*   Generic transmission request routing.
*
*   @description Routes transmission requests from higher level to lower level
*                modules.
*
*
*   @parameters
*       @param TxPduId   : ID of I-PDU to be transmitted
*              PduInfoPtr: Pointer to data length and pointer to I-SDU buffer
*              PDUR_SID  : Service ID of the calling module (e.g. for COM)
*
*   @return E_OK: Transmit request has been accepted
*           E_NOT_OK: Transmit request has not been accepted
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
#ifdef PDUR_ENABLE_PROFILING
static inline Std_ReturnType PduR_Transmit_PROFILED(PduIdType TxPduId, const PduInfoType* PduInfoPtr, const uint8 PDUR_SID);

static Std_ReturnType PduR_Transmit(PduIdType TxPduId, const PduInfoType* PduInfoPtr, const uint8 PDUR_SID)
{
    Std_ReturnType rval;

    dsProfilerWriteEvent(PduR_Transmit_PROFILER_SRC, UserTimespanStart, 0, TxPduId);
    rval = PduR_Transmit_PROFILED(TxPduId, PduInfoPtr, PDUR_SID);
    dsProfilerWriteEvent(PduR_Transmit_PROFILER_SRC, UserTimespanEnd, 0, 0);

    return rval;
}

static inline Std_ReturnType PduR_Transmit_PROFILED(PduIdType TxPduId, const PduInfoType* PduInfoPtr, const uint8 PDUR_SID)
#else
static Std_ReturnType PduR_Transmit(PduIdType TxPduId, const PduInfoType* PduInfoPtr, const uint8 PDUR_SID)
#endif
{
    /* Set the return value as E_OK */
    Std_ReturnType ReturnValue = E_OK;

    PduRTxRouting_Type* p_TxRoutingPaths;
    PduRRoutingPoint_Type* p_TxEndPoints;
    PduIdType DestPduRef;
    uint8 ModuleIndex;
    uint32 DestinationIndex;
    int i;
    INTENDED_UNREFERENCED_PARAMETER(PDUR_SID);

#if (PDUR_DEV_ERROR_DETECT == STD_ON)
    if (g_PduRState == PDUR_UNINIT)
    {
        /* Module not initialized */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_INVALID_REQUEST);
        ReturnValue = E_NOT_OK;
    }
    else if (TxPduId >= gp_PduRConfig->MaxNumTxPaths)
    {
        /* Invalid ID */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_PDU_ID_INVALID);
        ReturnValue = E_NOT_OK;
    }
    else
    {
        /* Do nothing */
    }
    if (ReturnValue == E_OK)
#endif
    {
        p_TxRoutingPaths = gp_PduRConfig->gp_TxRoutingPaths;
        p_TxEndPoints = gp_PduRConfig->gp_TxEndPoints;
        DestinationIndex = p_TxRoutingPaths[TxPduId].DestinationIndex;

        if (p_TxRoutingPaths[TxPduId].DestinationCount > 0)
        {
            /* PduR reports E_OK for a transmit request from an upper layer if at least one destination lower layer reports E_OK */
            ReturnValue = E_NOT_OK;
            for (i = 0; i < p_TxRoutingPaths[TxPduId].DestinationCount; i++)
            {
                ModuleIndex = p_TxEndPoints[DestinationIndex + i].ModuleIndex;
                DestPduRef = p_TxEndPoints[DestinationIndex + i].PduRef;

                if ((ModuleIndex != PDUR_TARGET_MODULE_NONE) && (0 != gp_PduRConfig->gp_TargetModules[ModuleIndex].transmitFunction))
                {
#ifdef PDUR_ENABLE_PROFILING
					DsTProfilerSource profSource = g_PduRProfilingSource[3*ModuleIndex];
					if (profSource != 0)
					{
						dsProfilerWriteEvent(profSource, UserTimespanStart, 3*ModuleIndex+1, TxPduId);
					}
#endif

                    /* PDUR201: Translate to lower layer ID */
                    if(gp_PduRConfig->gp_TargetModules[ModuleIndex].transmitFunction(DestPduRef, PduInfoPtr) == E_OK)
                    {
                        ReturnValue = E_OK;
                    }

#ifdef PDUR_ENABLE_PROFILING
					if (profSource != 0)
					{
						dsProfilerWriteEvent(profSource, UserTimespanEnd, 3*ModuleIndex+1, TxPduId);
					}
#endif

                }
            }
        }
    }
    return(ReturnValue);
}


/** <!------------------------------------------------------------------------->
*   Generic tx confirmation routing.
*
*   @description Confirmation from lower level interfaces after a PDU has been transmitted.
*
*
*   @parameters
*       @param TxPduId : ID of the PDU that has been transmitted.
*              PDUR_SID: Service ID of the calling module (e.g. for CAN, LIN, FR, IPDUM)
*
*   @return void
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
#ifdef _MSC_VER
#pragma warning (push)
#pragma warning (disable: 4505) //UNUSED
#endif
static void UNUSED PduR_TxConfirmation(PduIdType TxPduId, const uint8 PDUR_SID)
{
    PduRTxRouting_Type* p_TxRoutingPaths;
    PduIdType SourcePduRef;
    uint8 ModuleIndex;

    INTENDED_UNREFERENCED_PARAMETER(PDUR_SID);

#if (PDUR_DEV_ERROR_DETECT == STD_ON)
    if (g_PduRState == PDUR_UNINIT)
    {
        /* Module not initialized */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_INVALID_REQUEST);
    }
    else if (TxPduId >= gp_PduRConfig->MaxNumTxPaths)
    {
        /* Invalid ID */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_PDU_ID_INVALID);
    }
    else
#endif
    {
#if (PDUR_SD_SUPPORT == STD_ON)
        if( (gp_PduRConfig->gp_TxRoutingPaths == NULL_PTR) || (TxPduId >= PDUR_NUM_TX_PATHS) )
        {
            // If there are no Tx routing paths defined exit function
            return;
        }
#endif
        p_TxRoutingPaths = gp_PduRConfig->gp_TxRoutingPaths;
        ModuleIndex = p_TxRoutingPaths[TxPduId].Source.ModuleIndex;
        SourcePduRef = p_TxRoutingPaths[TxPduId].Source.PduRef;

#if (PDUR_GATEWAY_OPERATION == STD_ON)
        /* Execute Gateway functionality here*/
#endif

        if ((ModuleIndex != PDUR_TARGET_MODULE_NONE) && (0 != gp_PduRConfig->gp_TargetModules[ModuleIndex].txConfirmationFunction))
        {
            gp_PduRConfig->gp_TargetModules[ModuleIndex].txConfirmationFunction(SourcePduRef);
        }
    }
}

/** <!------------------------------------------------------------------------->
*   Generic trigger transmit routing
*
*   @description The function PduR_TriggerTransmit is called by lower layer modules
*                in case of bottom-up transmit requests. 
*
*   @parameters
*       @param TxPduId   : ID of PDU to be transmitted.
*              PduInfoPtr: Contains a pointer to a buffer (SduDataPtr) to where the SDU shall be copied to.
*                          On return, the service will indicate the length of the copied SDU data in SduLength.
*              PDUR_SID  : Service ID of calling module (e.g. for LIN, FR, IPDUM).
*
*   @return E_OK: Request has been accepted
*           E_NOT_OK: Request has not been accepted
*
*   @note
*      
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
static Std_ReturnType UNUSED PduR_TriggerTransmit(PduIdType TxPduId, PduInfoType* PduInfoPtr, const uint8 PDUR_SID)
{
    /* Set the return value as E_OK */
    Std_ReturnType ReturnValue = E_OK;

    PduRTxRouting_Type* p_TxRoutingPaths;
    PduIdType SourcePduRef;
    uint8 ModuleIndex;
    INTENDED_UNREFERENCED_PARAMETER(PDUR_SID);

#if (PDUR_DEV_ERROR_DETECT == STD_ON)
    if (g_PduRState == PDUR_UNINIT)
    {
        /* Module not initialized */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_INVALID_REQUEST);
        ReturnValue = E_NOT_OK;
    }
    else if (TxPduId >= gp_PduRConfig->MaxNumTxPaths)
    {
        /* Invalid ID */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_PDU_ID_INVALID);
        ReturnValue = E_NOT_OK;
    }
    else
    {
        /* Do nothing */
    }
    if (ReturnValue == E_OK)
#endif
    {
        p_TxRoutingPaths = gp_PduRConfig->gp_TxRoutingPaths;
        ModuleIndex = p_TxRoutingPaths[TxPduId].Source.ModuleIndex;
        SourcePduRef = p_TxRoutingPaths[TxPduId].Source.PduRef;

        if ((ModuleIndex != PDUR_TARGET_MODULE_NONE) && (0 != gp_PduRConfig->gp_TargetModules[ModuleIndex].triggerTransmitFunction))
        {
#ifdef PDUR_ENABLE_PROFILING
			DsTProfilerSource profSource = g_PduRProfilingSource[3*ModuleIndex+1];
			if (profSource != 0)
			{
				dsProfilerWriteEvent(profSource, UserTimespanStart, 3*ModuleIndex+2, TxPduId);
			}
#endif

            ReturnValue = gp_PduRConfig->gp_TargetModules[ModuleIndex].triggerTransmitFunction(SourcePduRef, PduInfoPtr);

#ifdef PDUR_ENABLE_PROFILING
			if (profSource != 0)
			{
				dsProfilerWriteEvent(profSource, UserTimespanEnd, 3*ModuleIndex+2, TxPduId);
			}
#endif

        }
        else
        {
            ReturnValue = E_NOT_OK;
        }
    }
    return ReturnValue;
}

/** <!------------------------------------------------------------------------->
*   Generic receive indication routing.
*
*   @description The function PduR_RxIndication is called by lower layer modules
*                after the PDU has been received.
*
*
*   @parameters
*       @param RxPduId:    Identifier of the PDU which has been received.
*              PduInfoPtr: Contains the length (SduLength) of the received I-PDU 
*                          and a pointer to a buffer (SduDataPtr) containing the I-PDU.
*              PDUR_SID:   Service ID of the calling module (e.g. for CAN, LIN, FR, IPDUM)
*
*   @return void
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
static void UNUSED PduR_RxIndication(PduIdType RxPduId, const PduInfoType* PduInfoPtr, const uint8 PDUR_SID)
{
    PduRRxRouting_Type* p_RxRoutingPaths;
    PduIdType DestPduRef;
    uint8 ModuleIndex;
    INTENDED_UNREFERENCED_PARAMETER(PDUR_SID);

#if (PDUR_DEV_ERROR_DETECT == STD_ON)
    if (g_PduRState == PDUR_UNINIT)
    {
        /* Module not initialized */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_INVALID_REQUEST);
    }
    else if (RxPduId >= gp_PduRConfig->MaxNumRxPaths)
    {
        /* Invalid ID */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_PDU_ID_INVALID);
    }
    else
#endif
    {
        p_RxRoutingPaths = gp_PduRConfig->gp_RxRoutingPaths;
        ModuleIndex = p_RxRoutingPaths[RxPduId].Destination.ModuleIndex;
        DestPduRef = p_RxRoutingPaths[RxPduId].Destination.PduRef;

#if (PDUR_GATEWAY_OPERATION == STD_ON)
        /* Execute Gateway functionality here*/
#endif
        if ((ModuleIndex != PDUR_TARGET_MODULE_NONE) && (0 != gp_PduRConfig->gp_TargetModules[ModuleIndex].rxIndicationFunction))
        {
#ifdef PDUR_ENABLE_PROFILING
			DsTProfilerSource profSource = g_PduRProfilingSource[3*ModuleIndex+2];
			if (profSource != 0)
			{
				dsProfilerWriteEvent(profSource, UserTimespanStart, 3*ModuleIndex+3, RxPduId);
			}
#endif

            /* PDUR420: PduId into the configured target PDU ID and route this indication to the configured upper layer module */
            gp_PduRConfig->gp_TargetModules[ModuleIndex].rxIndicationFunction(DestPduRef, PduInfoPtr);

#ifdef PDUR_ENABLE_PROFILING
			if (profSource != 0)
			{
				dsProfilerWriteEvent(profSource, UserTimespanEnd, 3*ModuleIndex+3, RxPduId);
			}
#endif

        }    
    }
}
#ifdef _MSC_VER
#pragma warning (pop)
#endif

/* PDUR407 */
#if (PDUR_COM_SUPPORT == STD_ON)
/** <!------------------------------------------------------------------------->
*   Requests a transmission for the AUTOSAR COM Module
*
*   @description Requests a tansmission for the AUTOSAR COM Module
*
*
*   @parameters
*       @param ComTxPduId: ID of AUTOSAR COM I-PDU to be transmitted
*              PduInfoPtr: Pointer to data length and pointer to I-SDU buffer
*
*   @return E_OK: Transmit request has been accepted
*           E_NOT_OK: Transmit request has not been accepted
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
Std_ReturnType PduR_ComTransmit(PduIdType TxPduId, const PduInfoType* PduInfoPtr)
{
    return PduR_Transmit(TxPduId, PduInfoPtr, PDUR_SID_COM_TRANSMIT);
}
#endif /* PDUR_COM_SUPPORT */

/* PDUR414,PDUR0778,PDUR417,PDUR422 */
#if (PDUR_IPDUM_SUPPORT == STD_ON)
/** <!------------------------------------------------------------------------->
*   Requests a transmission for the IPDU Multiplexer Module
*
*   @description Requests a transmission for the IPDU Multiplexer Module
*
*
*   @parameters
*       @param IpduMTxPduId: ID of IPDUM I-PDU to be transmitted
*              PduInfoPtr: Pointer to data length and pointer to I-SDU buffer
*
*   @return E_OK: Transmit request has been accepted
*           E_NOT_OK: Transmit request has not been accepted
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
Std_ReturnType PduR_IpduMTransmit(PduIdType TxPduId, const PduInfoType* PduInfoPtr)
{
    return PduR_Transmit(TxPduId, PduInfoPtr, PDUR_SID_IPDUM_TRANSMIT);
}


/** <!------------------------------------------------------------------------->
*   IpduM requests the buffer of the SDU for transmission from the PduR
*
*   @description IpduM requests the buffer of the SDU for transmission from the
*                PduR
*
*
*   @parameters
*       @param IpduMLoTxPduId: ID of IPDUM I-PDU to be transmitted
*              PduInfoPtr: Pointer to data length and pointer to I-SDU buffer
*
*   @return E_OK: Transmit request has been accepted
*           E_NOT_OK: Transmit request has not been accepted
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
Std_ReturnType PduR_IpduMTriggerTransmit(PduIdType TxPduId, PduInfoType* PduInfoPtr)
{
    return PduR_TriggerTransmit(TxPduId, PduInfoPtr, PDUR_SID_IPDUM_TRIGGER_TRANSMIT);
}

/** <!------------------------------------------------------------------------->
*   Transmission for the IPDU Multiplexer Module
*
*   @description Transmission for the IPDU Multiplexer Module
*
*
*   @parameters
*       @param IpduMLoTxPduId: ID of IPDUM I-PDU to be transmitted
*              
*
*   @return none
*           
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_IpduMTxConfirmation(PduIdType TxPduId)
{
    PduR_TxConfirmation(TxPduId, PDUR_SID_IPDUM_TX_CONFIRMATION);
}

/** <!------------------------------------------------------------------------->
*   Rx indication for the IpduM
*
*   @description The function PduR_IpduMRxIndication is called by the IpduM 
*                (acting as a lower layer module) after the PDU has been received
*
*   @parameters
*       @param IpduMLoRxPduId: ID of IpduM I-PDU that has been received.
*              PduInfoPtr    : Contains the length (SduLength) of the received I-PDU 
*                              and a pointer to a buffer (SduDataPtr) containing the I-PDU.      
*
*   @return none
*           
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_IpduMRxIndication(PduIdType RxPduId, const PduInfoType* PduInfoPtr)
{
    PduR_RxIndication(RxPduId, PduInfoPtr, PDUR_SID_IPDUM_RX_INDICATION);
}

#endif /* PDUR_IPDUM_SUPPORT */

#if (PDUR_SECOC_SUPPORT == STD_ON)
/** <!------------------------------------------------------------------------->
*   Requests a transmission for the SecOC Module
*
*   @description Requests a transmission for the SecOC Module
*
*
*   @parameters
*       @param SecOCTxPduId: ID of SecOC I-PDU to be transmitted
*              PduInfoPtr: Pointer to data length and pointer to I-SDU buffer
*
*   @return E_OK: Transmit request has been accepted
*           E_NOT_OK: Transmit request has not been accepted
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
Std_ReturnType PduR_SecOCTransmit(PduIdType TxPduId, const PduInfoType* PduInfoPtr)
{
    return PduR_Transmit(TxPduId, PduInfoPtr, PDUR_SID_SECOC_TRANSMIT);
}


/** <!------------------------------------------------------------------------->
*   SecOC requests the buffer of the SDU for transmission from the PduR
*
*   @description SecOC requests the buffer of the SDU for transmission from the
*                PduR
*
*
*   @parameters
*       @param TxPduId: ID of SecOC I-PDU to be transmitted
*              PduInfoPtr: Pointer to data length and pointer to I-SDU buffer
*
*   @return E_OK: Transmit request has been accepted
*           E_NOT_OK: Transmit request has not been accepted
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
Std_ReturnType PduR_SecOCTriggerTransmit(PduIdType TxPduId, PduInfoType* PduInfoPtr)
{
    return PduR_TriggerTransmit(TxPduId, PduInfoPtr, PDUR_SID_SECOC_TRIGGER_TRANSMIT);
}

/** <!------------------------------------------------------------------------->
*   Transmission for the SecOC Module
*
*   @description Transmission for the SecOC Module
*
*
*   @parameters
*       @param TxPduId: ID of SecOC I-PDU to be transmitted
*              
*
*   @return none
*           
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_SecOCTxConfirmation(PduIdType TxPduId)
{
    PduR_TxConfirmation(TxPduId, PDUR_SID_SECOC_TX_CONFIRMATION);
}

/** <!------------------------------------------------------------------------->
*   Rx indication for the SecOC
*
*   @description The function PduR_SecOCRxIndication is called by the SecOC 
*                (acting as a lower layer module) after the PDU has been received
*
*   @parameters
*       @param RxPduId      :   ID of SecOC I-PDU that has been received.
*              PduInfoPtr   :   Contains the length (SduLength) of the received I-PDU
*                               and a pointer to a buffer (SduDataPtr) containing the I-PDU.
*
*   @return none
*           
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_SecOCRxIndication(PduIdType RxPduId, const PduInfoType* PduInfoPtr)
{
    PduR_RxIndication(RxPduId, PduInfoPtr, PDUR_SID_SECOC_RX_INDICATION);
}

#endif /* PDUR_SECOC_SUPPORT */


#ifdef _MSC_VER
#pragma warning (push)
#pragma warning (disable: 4505) //UNUSED
#endif

/** <!------------------------------------------------------------------------->
*   Provides the received data of an I-PDU segment (N-PDU) to the upper layer
*
*   @description Provides the received data of an I-PDU segment (N-PDU) to the upper layer.
*                Each call to this function provides the next part of the I-PDU data.
*                The size of the remaining buffer is written to the position indicated by bufferSizePtr.
*
*
*   @parameters
*       @param id: Identification of the received I-PDU
*              info: Provides the source buffer (SduDataPtr) and the number of bytes to be copied (SduLength). 
*                    An SduLength of 0 can be used to query the current amount of available buffer in the upper layer module.
*                    In this case, the SduDataPtr may be a NULL_PTR.
*              bufferSizePtr: Available receive buffer after data has been copied
*              PDUR_SID  : Service ID of the calling module
*
*   @return BUFREQ_OK: Data copied successfully
*           BUFREQ_E_NOT_OK: Data was not copied because an error occurred
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
static BufReq_ReturnType UNUSED PduR_TpCopyRxData(PduIdType id, const PduInfoType* info, PduLengthType* bufferSizePtr, const uint8 PDUR_SID)
{
    INTENDED_UNREFERENCED_PARAMETER(PDUR_SID);

#if (PDUR_DEV_ERROR_DETECT == STD_ON)
    if (g_PduRState == PDUR_UNINIT)
    {
        /* Module not initialized */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_INVALID_REQUEST);
        return BUFREQ_E_NOT_OK;
    }
    else if (id >= gp_PduRConfig->MaxNumRxPaths)
    {
        /* Invalid ID */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_PDU_ID_INVALID);
        return BUFREQ_E_NOT_OK;
    }
    else if (info == NULL_PTR || bufferSizePtr == NULL_PTR)
    {
        /* Report DET error for a null pointer input */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_DATA_PTR_INVALID);
        return BUFREQ_E_NOT_OK;
    }
    else
#endif
    {
        PduRRxRouting_Type* p_RxRoutingPaths = gp_PduRConfig->gp_RxRoutingPaths;
        uint8 ModuleIndex = p_RxRoutingPaths[id].Destination.ModuleIndex;
        PduIdType DestPduRef = p_RxRoutingPaths[id].Destination.PduRef;

        if ((ModuleIndex != PDUR_TARGET_MODULE_NONE) && (0 != gp_PduRConfig->gp_TargetModules[ModuleIndex].copyRxDataFunction))
        {
            return gp_PduRConfig->gp_TargetModules[ModuleIndex].copyRxDataFunction(DestPduRef, info, bufferSizePtr);
        }
    }

    return BUFREQ_E_NOT_OK;
}


/** <!------------------------------------------------------------------------->
*   Acquires the transmit data of an I-PDU segment (N-PDU).
*
*   @description This function is called to acquire the transmit data of an I-PDU segment (N-PDU). Each call to this function
*                provides the next part of the I-PDU data unless retry->TpDataState is TP_DATARETRY. In this case the function
*                restarts to copy the data beginning at the offset from the current position indicated by retry->TxTpDataCnt.
*                The size of the remaining data is written to the position indicated by availableDataPtr.
*
*   @parameters
*       @param id: Identification of the transmitted I-PDU.
*              info: Provides the destination buffer (SduDataPtr) and the number of bytes to be copied (SduLength).
*                    If not enough transmit data is available, no data is copied by the upper layer module and BUFREQ_E_BUSY is returned.
*                    The lower layer module may retry the call. An SduLength of 0 can be used to indicate state changes in the retry parameter
*                    or to query the current amount of available data in the upper layer module. In this case, the SduDataPtr may be a NULL_PTR.
*              retry: This parameter is used to acknowledge transmitted data or to retransmit data after transmission problems.
*                     If the retry parameter is a NULL_PTR, it indicates that the transmit data can be removed from the buffer immediately after it has been copied.
*                     Otherwise, the retry parameter must point to a valid RetryInfoType element.
*                     If TpDataState indicates TP_CONFPENDING, the previously copied data must remain in the TP buffer to be available for error recovery.
*                     TP_DATACONF indicates that all data that has been copied before this call is confirmed and can be removed from the TP buffer.
*                     Data copied by this API call is excluded and will be confirmed later.
*                     TP_DATARETRY indicates that this API call shall copy previously copied data in order to recover from an error.
*                     In this case TxTpDataCnt specifies the offset in bytes from the current data copy position.
*              availableDataPtr: Indicates the remaining number of bytes that are available in the upper layer module's Tx buffer.
*              PDUR_SID  : Service ID of the calling module
*
*   @return BUFREQ_OK: Data has been copied to the transmit buffer completely as requested.
*           BUFREQ_E_BUSY: Request could not be fulfilled, because the required amount of Tx data is not available.
*                          The lower layer module may retry this call later on. No data has been copied.
*           BUFREQ_E_NOT_OK: Data has not been copied. Request failed.
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
static BufReq_ReturnType UNUSED PduR_TpCopyTxData(PduIdType id, const PduInfoType* info, const RetryInfoType* retry, PduLengthType* availableDataPtr, const uint8 PDUR_SID)
{
    INTENDED_UNREFERENCED_PARAMETER(PDUR_SID);

#if (PDUR_DEV_ERROR_DETECT == STD_ON)
    if (g_PduRState == PDUR_UNINIT)
    {
        /* Module not initialized */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_INVALID_REQUEST);
        return BUFREQ_E_NOT_OK;
    }
    else if (id >= gp_PduRConfig->MaxNumTxPaths)
    {
        /* Invalid ID */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_PDU_ID_INVALID);
        return BUFREQ_E_NOT_OK;
    }
    else if (info == NULL_PTR || availableDataPtr == NULL_PTR)
    {
        /* Report DET error for a null pointer input */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_DATA_PTR_INVALID);
        return BUFREQ_E_NOT_OK;
    }
    else
#endif
    {
        PduRTxRouting_Type* p_TxRoutingPaths = gp_PduRConfig->gp_TxRoutingPaths;
        uint8 ModuleIndex = p_TxRoutingPaths[id].Source.ModuleIndex;
        PduIdType SourcePduRef = p_TxRoutingPaths[id].Source.PduRef;

        if ((ModuleIndex != PDUR_TARGET_MODULE_NONE) && (0 != gp_PduRConfig->gp_TargetModules[ModuleIndex].copyTxDataFunction))
        {
            return gp_PduRConfig->gp_TargetModules[ModuleIndex].copyTxDataFunction(SourcePduRef, info, retry, availableDataPtr);
        }
    }

    return BUFREQ_E_NOT_OK;
}

/** <!------------------------------------------------------------------------->
*   Start of receiving an N-SDU.
*
*   @description This function is called at the start of receiving an N-SDU.
*                The N-SDU might be fragmented into multiple N-PDUs (FF with one or more following CFs) or might consist of a single N-PDU (SF).
*                The service shall provide the currently available maximum buffer size when invoked with TpSduLength equal to 0.
*
*   @parameters
*       @param id: Identification of the transmitted I-PDU.
*              info: Pointer to a PduInfoType structure containing the payload data (without protocol information) and payload length of the first 
*                    frame or single frame of a transport protocol I-PDU reception, and the MetaData related to this PDU.
*                    If neither first/single frame data nor MetaData are available, this parameter is set to NULL_PTR.
*              TpSduLength: Total length of the N-SDU to be received.
*              bufferSizePtr: Available receive buffer in the receiving module.
*                             This parameter will be used to compute the Block Size (BS) in the transport protocol module.
*              PDUR_SID  : Service ID of the calling module
*
*   @return BUFREQ_OK: Connection has been accepted. bufferSizePtr indicates the available receive buffer; reception is continued.
*                      If no buffer of the requested size is available, a receive buffer size of 0 shall be indicated by bufferSizePtr.
*           BUFREQ_E_NOT_OK: Connection has been rejected; reception is aborted. bufferSizePtr remains unchanged.
*           BUFREQ_E_OVFL: No buffer of the required length can be provided; reception is aborted. bufferSizePtr remains unchanged.
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
static BufReq_ReturnType UNUSED PduR_TpStartOfReception(PduIdType id, const PduInfoType* info, PduLengthType TpSduLength, PduLengthType* bufferSizePtr, const uint8 PDUR_SID)
{
    INTENDED_UNREFERENCED_PARAMETER(PDUR_SID);

#if (PDUR_DEV_ERROR_DETECT == STD_ON)
    if (g_PduRState == PDUR_UNINIT)
    {
        /* Module not initialized */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_INVALID_REQUEST);
        return BUFREQ_E_NOT_OK;
    }
    else if (id >= gp_PduRConfig->MaxNumRxPaths)
    {
        /* Invalid ID */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_PDU_ID_INVALID);
        return BUFREQ_E_NOT_OK;
    }
    else if (info == NULL_PTR || bufferSizePtr == NULL_PTR)
    {
        /* Report DET error for a null pointer input */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_DATA_PTR_INVALID);
        return BUFREQ_E_NOT_OK;
    }
    else
#endif
    {
        PduRRxRouting_Type* p_RxRoutingPaths = gp_PduRConfig->gp_RxRoutingPaths;
        uint8 ModuleIndex = p_RxRoutingPaths[id].Destination.ModuleIndex;
        PduIdType DestPduRef = p_RxRoutingPaths[id].Destination.PduRef;

        if ((ModuleIndex != PDUR_TARGET_MODULE_NONE) && (0 != gp_PduRConfig->gp_TargetModules[ModuleIndex].startOfReceptionFunction))
        {
            return gp_PduRConfig->gp_TargetModules[ModuleIndex].startOfReceptionFunction(DestPduRef, info, TpSduLength, bufferSizePtr);
        }
    }

    return BUFREQ_E_NOT_OK;
}

/** <!------------------------------------------------------------------------->
*   Transmission Confirmation
*
*   @description This function is called after the I-PDU has been transmitted on its network, 
*                the result indicates whether the transmission was successful or not.
*
*
*   @parameters
*       @param id: Identification of the transmitted I-PDU.
*              result: Result of the transmission of the I-PDU.
*              PDUR_SID  : Service ID of the calling module
*
*
*   @return none
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
static void UNUSED PduR_TpTxConfirmation(PduIdType id, Std_ReturnType result, const uint8 PDUR_SID)
{
    INTENDED_UNREFERENCED_PARAMETER(PDUR_SID);

#if (PDUR_DEV_ERROR_DETECT == STD_ON)
    if (g_PduRState == PDUR_UNINIT)
    {
        /* Module not initialized */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_INVALID_REQUEST);
    }
    else if (id >= gp_PduRConfig->MaxNumTxPaths)
    {
        /* Invalid ID */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_PDU_ID_INVALID);
    }
    else
#endif
    {
        PduRTxRouting_Type* p_TxRoutingPaths = gp_PduRConfig->gp_TxRoutingPaths;
        uint8 ModuleIndex = p_TxRoutingPaths[id].Source.ModuleIndex;
        PduIdType SourcePduRef = p_TxRoutingPaths[id].Source.PduRef;

        if ((ModuleIndex != PDUR_TARGET_MODULE_NONE) && (0 != gp_PduRConfig->gp_TargetModules[ModuleIndex].tpTxConfirmationFunction))
        {
            gp_PduRConfig->gp_TargetModules[ModuleIndex].tpTxConfirmationFunction(SourcePduRef, result);
        }
    }
}

/** <!------------------------------------------------------------------------->
*   Reception Indication
*
*   @description This function is called after an I-PDU has been received via the TP API,
*                the result indicates whether the transmission was successful or not.
*
*
*   @parameters
*       @param id: Identification of the transmitted I-PDU.
*              result: Result of the transmission of the I-PDU.
*              PDUR_SID  : Service ID of the calling module
*
*
*   @return none
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
static void UNUSED PduR_TpRxIndication(PduIdType id, Std_ReturnType result, const uint8 PDUR_SID)
{
    INTENDED_UNREFERENCED_PARAMETER(PDUR_SID);

#if (PDUR_DEV_ERROR_DETECT == STD_ON)
    if (g_PduRState == PDUR_UNINIT)
    {
        /* Module not initialized */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_INVALID_REQUEST);
    }
    else if (id >= gp_PduRConfig->MaxNumRxPaths)
    {
        /* Invalid ID */
        Det_ReportError (PDUR_MODULE_ID,
                         PDUR_INSTANCE_ID,
                         PDUR_SID,
                         PDUR_E_PDU_ID_INVALID);
    }
    else
#endif
    {
        PduRRxRouting_Type* p_RxRoutingPaths = gp_PduRConfig->gp_RxRoutingPaths;
        uint8 ModuleIndex = p_RxRoutingPaths[id].Destination.ModuleIndex;
        PduIdType DestPduRef = p_RxRoutingPaths[id].Destination.PduRef;

        if ((ModuleIndex != PDUR_TARGET_MODULE_NONE) && (0 != gp_PduRConfig->gp_TargetModules[ModuleIndex].tpRxIndicationFunction))
        {
            gp_PduRConfig->gp_TargetModules[ModuleIndex].tpRxIndicationFunction(DestPduRef, result);
        }
    }
}

#ifdef _MSC_VER
#pragma warning (pop)
#endif

#if (PDUR_J1939TP_SUPPORT == STD_ON)

/** <!------------------------------------------------------------------------->
*   Provides the received data of an I-PDU segment (N-PDU) to the upper layer
*
*   @description Provides the received data of an I-PDU segment (N-PDU) to the upper layer.
*                Each call to this function provides the next part of the I-PDU data.
*                The size of the remaining buffer is written to the position indicated by bufferSizePtr.
*
*
*   @parameters
*       @param id: Identification of the received I-PDU
*              info: Provides the source buffer (SduDataPtr) and the number of bytes to be copied (SduLength). 
*                    An SduLength of 0 can be used to query the current amount of available buffer in the upper layer module.
*                    In this case, the SduDataPtr may be a NULL_PTR.
*              bufferSizePtr: Available receive buffer after data has been copied
*
*   @return BUFREQ_OK: Data copied successfully
*           BUFREQ_E_NOT_OK: Data was not copied because an error occurred
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
BufReq_ReturnType PduR_J1939TpCopyRxData(PduIdType id, const PduInfoType* info, PduLengthType* bufferSizePtr)
{
    return PduR_TpCopyRxData(id, info, bufferSizePtr, PDUR_SID_J1939TP_COPY_RX_DATA);
}


/** <!------------------------------------------------------------------------->
*   Acquires the transmit data of an I-PDU segment (N-PDU).
*
*   @description This function is called to acquire the transmit data of an I-PDU segment (N-PDU). Each call to this function
*                provides the next part of the I-PDU data unless retry->TpDataState is TP_DATARETRY. In this case the function
*                restarts to copy the data beginning at the offset from the current position indicated by retry->TxTpDataCnt.
*                The size of the remaining data is written to the position indicated by availableDataPtr.
*
*   @parameters
*       @param id: Identification of the transmitted I-PDU.
*              info: Provides the destination buffer (SduDataPtr) and the number of bytes to be copied (SduLength).
*                    If not enough transmit data is available, no data is copied by the upper layer module and BUFREQ_E_BUSY is returned.
*                    The lower layer module may retry the call. An SduLength of 0 can be used to indicate state changes in the retry parameter
*                    or to query the current amount of available data in the upper layer module. In this case, the SduDataPtr may be a NULL_PTR.
*              retry: This parameter is used to acknowledge transmitted data or to retransmit data after transmission problems.
*                     If the retry parameter is a NULL_PTR, it indicates that the transmit data can be removed from the buffer immediately after it has been copied.
*                     Otherwise, the retry parameter must point to a valid RetryInfoType element.
*                     If TpDataState indicates TP_CONFPENDING, the previously copied data must remain in the TP buffer to be available for error recovery.
*                     TP_DATACONF indicates that all data that has been copied before this call is confirmed and can be removed from the TP buffer.
*                     Data copied by this API call is excluded and will be confirmed later.
*                     TP_DATARETRY indicates that this API call shall copy previously copied data in order to recover from an error.
*                     In this case TxTpDataCnt specifies the offset in bytes from the current data copy position.
*              availableDataPtr: Indicates the remaining number of bytes that are available in the upper layer module's Tx buffer.
*
*   @return BUFREQ_OK: Data has been copied to the transmit buffer completely as requested.
*           BUFREQ_E_BUSY: Request could not be fulfilled, because the required amount of Tx data is not available.
*                          The lower layer module may retry this call later on. No data has been copied.
*           BUFREQ_E_NOT_OK: Data has not been copied. Request failed.
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
BufReq_ReturnType PduR_J1939TpCopyTxData(PduIdType id, const PduInfoType* info, const RetryInfoType* retry, PduLengthType* availableDataPtr)
{
    return PduR_TpCopyTxData(id, info, retry, availableDataPtr, PDUR_SID_J1939TP_COPY_TX_DATA);
}

/** <!------------------------------------------------------------------------->
*   Start of receiving an N-SDU.
*
*   @description This function is called at the start of receiving an N-SDU.
*                The N-SDU might be fragmented into multiple N-PDUs (FF with one or more following CFs) or might consist of a single N-PDU (SF).
*                The service shall provide the currently available maximum buffer size when invoked with TpSduLength equal to 0.
*
*   @parameters
*       @param id: Identification of the transmitted I-PDU.
*              info: Pointer to a PduInfoType structure containing the payload data (without protocol information) and payload length of the first 
*                    frame or single frame of a transport protocol I-PDU reception, and the MetaData related to this PDU.
*                    If neither first/single frame data nor MetaData are available, this parameter is set to NULL_PTR.
*              TpSduLength: Total length of the N-SDU to be received.
*              bufferSizePtr: Available receive buffer in the receiving module.
*                             This parameter will be used to compute the Block Size (BS) in the transport protocol module.
*
*   @return BUFREQ_OK: Connection has been accepted. bufferSizePtr indicates the available receive buffer; reception is continued.
*                      If no buffer of the requested size is available, a receive buffer size of 0 shall be indicated by bufferSizePtr.
*           BUFREQ_E_NOT_OK: Connection has been rejected; reception is aborted. bufferSizePtr remains unchanged.
*           BUFREQ_E_OVFL: No buffer of the required length can be provided; reception is aborted. bufferSizePtr remains unchanged.
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
BufReq_ReturnType PduR_J1939TpStartOfReception(PduIdType id, const PduInfoType* info, PduLengthType TpSduLength, PduLengthType* bufferSizePtr)
{
    return PduR_TpStartOfReception(id, info, TpSduLength, bufferSizePtr, PDUR_SID_J1939TP_START_OF_RECEPTION);
}

/** <!------------------------------------------------------------------------->
*   Transmission Confirmation
*
*   @description This function is called after the I-PDU has been transmitted on its network, 
*                the result indicates whether the transmission was successful or not.
*
*
*   @parameters
*       @param id: Identification of the transmitted I-PDU.
*              result: Result of the transmission of the I-PDU.
*
*
*   @return none
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_J1939TpTxConfirmation(PduIdType id, Std_ReturnType result)
{
    PduR_TpTxConfirmation(id, result, PDUR_SID_J1939TP_TX_CONFIRMATION);
}

/** <!------------------------------------------------------------------------->
*   Reception Indication
*
*   @description This function is called after an I-PDU has been received via the TP API,
*                the result indicates whether the transmission was successful or not.
*
*
*   @parameters
*       @param id: Identification of the transmitted I-PDU.
*              result: Result of the transmission of the I-PDU.
*
*
*   @return none
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_J1939TpRxIndication(PduIdType id, Std_ReturnType result)
{
    PduR_TpRxIndication(id, result, PDUR_SID_J1939TP_RX_INDICATION);
}

/** <!------------------------------------------------------------------------->
*   Trigger transmit for the J1939Tp Interface
*
*   @description The function PduR_J1939TpTriggerTransmit is called by the J1939Tp module
                 for re-transmitting a PDU.
*
*   @parameters
*       @param J1939TpTxPduId : ID of J1939 PDU that has been transmitted.
*              PduInfoPtr: Contains a pointer to a buffer (SduDataPtr) to where the SDU shall be copied to.
*                          On return, the service will indicate the length of the copied SDU data in SduLength
*
*   @return E_OK:     Transmit request has been accepted
*           E_NOT_OK: Transmit request has not been accepted
*
*   @note
*      Contrary to: Specification of a Transport Layer for SAE J1939 AUTOSAR CP Release 4.4.0 (4.1 Limitations)
*                   The J1939Tp module does not implement the TriggerTransmit API, because it is only needed for time triggered bus architectures.
*      the Bus Manager-J1939Requirements and -Architecture requires a TriggerTransmit function to automatically re-transmitting a J1939-Pdu.
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
Std_ReturnType PduR_J1939TpTriggerTransmit(PduIdType TxPduId, PduInfoType* PduInfoPtr)
{
    return PduR_TriggerTransmit(TxPduId, PduInfoPtr, PDUR_SID_J1939TP_TRIGGER_TRANSMIT);
}

#endif /* PDUR_J1939TP_SUPPORT */



/* PDUR349,PDUR345 */
#if (PDUR_CANIF_SUPPORT  == STD_ON)
/** <!------------------------------------------------------------------------->
*   Tx confirmation from the CAN Interface
*
*   @description Confirmation from the CAN Interface after the PDU has been 
*                transmitted on the CAN network
*
*
*   @parameters
*       @param CanTxPduId: ID of CAN L-PDU that has been transmitted
*
*   @return void
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_CanIfTxConfirmation(PduIdType TxPduId)
{
    PduR_TxConfirmation(TxPduId, PDUR_SID_CANIF_TX_CONFIRMATION);
}

/** <!------------------------------------------------------------------------->
*   Rx Indication for the CAN Interface
*
*   @description The function PduR_CanIfRxIndication is called by the CAN Interface 
*                after a CAN L-PDU has been received.
*
*
*   @parameters
*       @param CanRxPduId: ID of CAN L-PDU that has been received.
*              PduInfoPtr: Contains the length (SduLength) of the received I-PDU 
*                          and a pointer to a buffer (SduDataPtr) containing the I-PDU.
*
*   @return void
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_CanIfRxIndication(PduIdType RxPduId, const PduInfoType* PduInfoPtr)
{
    PduR_RxIndication(RxPduId, PduInfoPtr, PDUR_SID_CANIF_RX_INDICATION);
}
#endif /* PDUR_CANIF_SUPPORT */

#if (PDUR_CANNM_SUPPORT == STD_ON)
/** <!------------------------------------------------------------------------->
*   Tx confirmation from the CAN NM
*
*   @description Confirmation from the CAN network management module
*
*
*   @parameters
*       @param CanNmTxPduId: ID of CAN L-PDU that has been transmitted
*
*   @return void
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_CanNmTxConfirmation(PduIdType TxPduId)
{
    PduR_TxConfirmation(TxPduId, PDUR_SID_CANNM_TX_CONFIRMATION);
}

/** <!------------------------------------------------------------------------->
*   Triggers the transmission of a CanNm Message
*
*   @description Triggers the transmission of a CanNm Message. Request for the
*                user data content of the NM message be sent
*
*   @parameters
*       @param CanNmTxPduId: ID of IPDUM I-PDU to be transmitted
*              PduInfoPtr: Pointer to data length and pointer to I-SDU buffer
*
*   @return E_OK: Transmit request has been accepted
*           E_NOT_OK: Transmit request has not been accepted
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
Std_ReturnType PduR_CanNmTriggerTransmit(PduIdType TxPduId, PduInfoType* PduInfoPtr)
{
    return PduR_TriggerTransmit(TxPduId, PduInfoPtr, PDUR_SID_CANNM_TRIGGER_TRANSMIT);
}

/** <!------------------------------------------------------------------------->
*   Rx Indication for CanNm
*
*   @description Reports the ERBA and IRBA to the COM
*
*   @parameters
*       @param CanNmRxPduId: ID of CAN L-PDU that has been received.
*              PduInfoPtr  : Pointer to data length and pointer to I-SDU buffer
*
*   @return None
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_CanNmRxIndication(PduIdType RxPduId, const PduInfoType* PduInfoPtr)
{
    PduR_RxIndication(RxPduId, PduInfoPtr, PDUR_SID_CANNM_RX_INDICATION);
}

#endif /* PDUR_CANNM_SUPPORT */

/* PDUR384,PDUR390 */
#if (PDUR_LINIF_SUPPORT  == STD_ON)
/** <!------------------------------------------------------------------------->
*   Tx confirmation from the LIN Interface
*
*   @description Confirmation from the LIN Interface after the PDU has been 
*                transmitted on the LIN network
*
*
*   @parameters
*       @param LinTxPduId: ID of LIN L-PDU that has been transmitted
*
*   @return void
*
*   @note
*
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_LinIfTxConfirmation(PduIdType TxPduId)
{
    PduR_TxConfirmation(TxPduId, PDUR_SID_LINIF_TX_CONFIRMATION);
}

/** <!------------------------------------------------------------------------->
*   Rx indication for the LIN Interface
*
*   @description The function PduR_LinIfRxIndication is called by the Lin Interface 
*                after a LIN L-PDU has been received.
*
*   @parameters
*       @param LinRxPduId: ID of LIN L-PDU that has been received.
*              PduInfoPtr: Contains the length (SduLength) of the received I-PDU and 
*                          a pointer to a buffer (SduDataPtr) containing the I-PDU. 
*   @return void
*
*   @note
*      
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
void PduR_LinIfRxIndication(PduIdType RxPduId, const PduInfoType* PduInfoPtr)
{
    PduR_RxIndication(RxPduId, PduInfoPtr, PDUR_SID_LINIF_RX_INDICATION);
}

/** <!------------------------------------------------------------------------->
*   Trigger transmit for the LIN Interface
*
*   @description The function PduR_LinIfTriggerTransmit is called by the Lin Interface 
                 for sending out a Lin frame.
*
*   @parameters
*       @param LinTxPduId : ID of Lin L-PDU that has been transmitted.
*              PduInfoPtr: Contains a pointer to a buffer (SduDataPtr) to where the SDU shall be copied to.
*                          On return, the service will indicate the length of the copied SDU data in SduLength
*
*   @return E_OK: Transmit request has been accepted
*           E_NOT_OK: Transmit request has not been accepted
*
*   @note
*      
*   @messages
*
*
*<!-------------------------------------------------------------------------->*/
Std_ReturnType PduR_LinIfTriggerTransmit(PduIdType TxPduId, PduInfoType* PduInfoPtr)
{
    return PduR_TriggerTransmit(TxPduId, PduInfoPtr, PDUR_SID_LINIF_TRIGGER_TRANSMIT);
}
#endif /* PDUR_LINIF_SUPPORT */

#if (PDUR_SOAD_SUPPORT == STD_ON)
void PduR_SoAdIfTxConfirmation(const SoAd_PduGlobalType* PduRRef, PduIdType TxPduId)
{
  (void)PduRRef;
  PduR_TxConfirmation(TxPduId, 0/*TODO*/);
}

void PduR_SoAdIfRxIndication(const SoAd_PduGlobalType* PduRRef, PduIdType RxPduId, PduInfoType *PduInfoPtr)
{
    if (PduRRef != 0)
    {
        if (PduRRef->PDUG_SoAdIfRxIndication != 0)
        {
            // needed for SoAd_SdRouter and SoAd_NmRouter
            PduRRef->PDUG_SoAdIfRxIndication(RxPduId, PduInfoPtr);
        }
        else
        {
            (void)PduRRef;
            PduR_RxIndication(RxPduId, PduInfoPtr, 0/*TODO*/);
        }
    }
}

Std_ReturnType PduR_SoAdIfTriggerTransmit(const SoAd_PduGlobalType* PduRRef, PduIdType TxPduId, PduInfoType* PduInfoPtr)
{
  (void)PduRRef;
  return PduR_TriggerTransmit(TxPduId, PduInfoPtr, 0/*TODO*/);
}

BufReq_ReturnType PduR_SoAdTpCopyRxData(const SoAd_PduGlobalType* PduRRef, PduIdType RxPduId, PduInfoType* RxPduInfo, PduLengthType* RxPduLength)
{
  // TODO Add implementation or remove
  (void)PduRRef;
  (void)RxPduId;
  (void)RxPduInfo;
  (void)RxPduLength;
  return BUFREQ_E_NOT_OK;
}

BufReq_ReturnType PduR_SoAdTpCopyTxData(const SoAd_PduGlobalType* PduRRef, PduIdType TxPduId, PduInfoType* TxPduInfo, RetryInfoType* TxRetryInfo, PduLengthType* TxPduLength)
{
  // TODO Add implementation or remove
  (void)PduRRef;
  (void)TxPduId;
  (void)TxPduInfo;
  (void)TxPduLength;
  return BUFREQ_E_NOT_OK;
}

void PduR_SoAdTpRxIndication(const SoAd_PduGlobalType* PduRRef, PduIdType RxPduId, NotifResultType result)
{
  // TODO Add implementation or remove
  (void)PduRRef;
  (void)RxPduId;
  (void)result;
  return;
}

BufReq_ReturnType PduR_SoAdTpStartOfReception(const SoAd_PduGlobalType* PduRRef, PduIdType RxPduId, PduInfoType *RxPduinfo, PduLengthType RxTpSduLength, PduLengthType *RxBufferSizePtr)
{
  // TODO Add implementation or remove
  (void)PduRRef;
  (void)RxPduId;
  (void)RxPduinfo;
  (void)RxTpSduLength;
  (void)RxBufferSizePtr;
  return BUFREQ_E_NOT_OK;
}

void PduR_SoAdTpTxConfirmation(const SoAd_PduGlobalType* PduRRef, PduIdType TxPduId, NotifResultType TxResult)
{
  // TODO Add implementation or remove
  (void)PduRRef;
  (void)TxPduId;
  (void)TxResult;
  return;
}

void PduR_SoConModeChg(const SoAd_BswModuleType *BswModuleRef, SoAd_SoConIdType SoConId, SoAd_SoConModeType Mode)
{
  (void)BswModuleRef;
  (void)SoConId;
  (void)Mode;

#if (PDUR_SD_SUPPORT == STD_ON)
    Sd_SoAdSoConModeChg(SoConId, Mode);
    if (BswModuleRef == 0) {
        return;
    }
    else {
        if (BswModuleRef->BsWm_SoAdSoConModeChg == 0) {
            return;
        }
        else {
            BswModuleRef->BsWm_SoAdSoConModeChg(SoConId, Mode);
        }
    }
#endif // (PDUR_SD_SUPPORT == STD_ON)
  return;
}

void PduR_LocalIpAddrAssignmentChg(const SoAd_BswModuleType *BswModuleRef, SoAd_SoConIdType SoConId, TcpIp_IpAddrStateType State)
{
  (void)BswModuleRef;
  (void)SoConId;
  (void)State;

#if (PDUR_SD_SUPPORT == STD_ON)
    Sd_LocalIpAddrAssignmentChg(SoConId, State);
    if (BswModuleRef == 0) {
        return;
    } else {
        if (BswModuleRef->BsWm_SoAdSoConModeChg == 0) {
            return;
        } else {
            BswModuleRef->BsWm_SoAdLocalIpAddrAssignmentChg(SoConId, State);
        }
    }
#endif // (PDUR_SD_SUPPORT == STD_ON)
  return;
}

#endif /* PDUR_SOAD_SUPPORT */

/*----------------------------------------------------------------------------*/
/* STATIC FUNCTIONS */
/*----------------------------------------------------------------------------*/

#if (PDUR_MULTICAST_TOIF_SUPPORT == STD_ON)
Std_ReturnType PduR_MultiIfTransmit(PduR_TransmitIfType* p_PduRXTx, PduIdType MPduId, const PduInfoType* PduInfoPtr)
{
    /* Set the return value as E_OK */
    Std_ReturnType ReturnValue = E_OK;
    Std_ReturnType IfReturn;
    PduR_MultiCastRoutngType* p_MultiCastTable;
    uint8 FuncIndex;
    uint8 LoopIndex;
    p_MultiCastTable = &p_PduRXTx->gp_MultiRouteTable[MPduId];

    for (LoopIndex = 0; LoopIndex < p_MultiCastTable->NumDestinations; LoopIndex++)
    {
        FuncIndex = p_MultiCastTable->gp_MultiCastTable[LoopIndex].TargetFuncIndex;
        /* PDUR201,237:Translate to lower layer ID */
        IfReturn = ((TransmitFunction)gp_PduRConfig->gp_TgtFunc[FuncIndex])(p_MultiCastTable->gp_MultiCastTable[LoopIndex].TargetPduId,PduInfoPtr);
        /* PDUR218,238: Multicast error */
        if (IfReturn != E_OK)
        {
            ReturnValue = E_NOT_OK;
        }
    }
    return ReturnValue;
}
#endif



#endif

